Livemap = {}
Livemap_mp = Class(Livemap)

function Livemap:new()
    local map = {
        xmlFileStatic = nil,
        xmlFileDynamic = nil,
        tickCount = nil,
        environment = nil,
        weather = nil,
        windUpdater = nil,
        forecast = nil,
        fieldCache = nil,
        hotSpotsCache = nil,
        placeableSystem = nil,
        lastStaticGenerateTick = nil,
        twister = nil,
        xmlVersion = nil,
        modVersion = nil
    }

    setmetatable(map, self)
    self.__index = self

    return map
end

function onStart()
    Livemap:onStart()
end

function Livemap:onStart()
    self.fieldCache = {}
    self.hotSpotsCache = {}
    self.environment = g_currentMission.environment
    self.weather = self.environment.weather
    self.windUpdater = self.weather.windUpdater
    self.forecast = self.weather.forecast
    self.placeableSystem = g_currentMission.placeableSystem
    self.lastPlayerCount = 0
    self.twister = self.weather.twister
    self.xmlVersion = 2
    self.modVersion = "1.1.0.2"

    if g_currentMission.connectedToDedicatedServer then
        self:generateFileStatic(true)
        self:generateFileDynamic()
        self:tick()
    else
        Utils:debugPrint("VG Livemap can be used only on a Dedicated Server")
    end
end

function Livemap:tick()
    self.tickCount = (self.tickCount or 0) + 1
    self.lastStaticGenerateTick = (self.lastStaticGenerateTick or 0) + 1

    if Utils:modulo(self.tickCount, 5) == 0 then
        if self.lastPlayerCount > 0 then
            self:generateFileDynamic()
        end

        self.lastPlayerCount = Utils:getPlayersCount()
    end

    if Utils:modulo(self.tickCount, 60) == 0 then
        if self.lastPlayerCount > 0 then
            self:generateFileStatic(false)
        end

        self.tickCount = 0
    end

    if self.lastStaticGenerateTick > 3600 * 23 then
        self:generateFileStatic(true)
        self.lastStaticGenerateTick = 0
    end

    addTimer(1 * 1000, "tick", self)
end

function Livemap:createXmlFile(name)
    local path = getUserProfileAppPath(); -- Server main path
    return XMLFile.create("Server", path .. "/modSettings/" .. name, "Server")
end

function Livemap:generateFileDynamic()
    local map = g_mapManager:getMapById(g_currentMission.missionInfo.mapId)

    if map == nil or map.title == nil then
        Utils:debugPrint("Unable to generate livemap_dynamic.xml, map.title is null")
    end

    self.xmlFileDynamic = self:createXmlFile("livemap_dynamic.xml")

    self.xmlFileDynamic:setInt("Server#version", self.xmlVersion)
    self.xmlFileDynamic:setInt("Server#gameVersion", 25)
    self.xmlFileDynamic:setString("Server#modVersion", self.modVersion)
    self.xmlFileDynamic:setString("Server#lastUpdate", getDate("%Y/%m/%d %H:%M:%S"))
    self.xmlFileDynamic:setBool("Server#paused", g_currentMission.paused)
    self.xmlFileDynamic:setString("Server#mapName", map.title)

    if g_currentMission.userManager ~= nil then
        self:savePlayers()
    end

    if g_currentMission.vehicleSystem ~= nil then
        self:saveVehicles()
    end

    if self.forecast ~= nil then
        self:saveWeatherHourly()
        self:saveWeatherDaily()
    end

    if self.twister ~= nil then
        self:saveTwister();
    end

    self:saveCalendar()

    self.xmlFileDynamic:save()
    self.xmlFileDynamic:delete()
end

function Livemap:generateFileStatic(bypassCache)
    local map = g_mapManager:getMapById(g_currentMission.missionInfo.mapId)

    if map == nil or map.title == nil then
        Utils:debugPrint("Unable to generate livemap_static.xml, map.title is null")
    end

    if g_fieldManager ~= nil then
        self.xmlFileStatic = self:createXmlFile("livemap_static.xml")

        local wroteFields = self:updateFields()
        local wroteHotSpot = self:updateHotSpots()

        if bypassCache == true or (bypassCache == false and (wroteFields == true or wroteHotSpot == true)) then
            self.xmlFileStatic:setInt("Server#version", self.xmlVersion)
            self.xmlFileStatic:setInt("Server#gameVersion", 25)
            self.xmlFileStatic:setString("Server#modVersion", self.modVersion)
            self.xmlFileStatic:setString("Server#mapName", map.title)
            self.xmlFileStatic:setInt("Server#size", g_currentMission.terrainSize);
            self:writeFields()
            self:writeHotSpots()

            self.xmlFileStatic:save()
            self.xmlFileStatic:delete()
        end
    end
end

function Livemap:saveCalendar()
    for index, fruitType in ipairs(g_fruitTypeManager:getFruitTypes()) do
        local key = "Server.Calendar";
        self.xmlFileDynamic:setFloat(key .. "#progress", self:getSeasonPercentageProgress())

        local fruitKey = key .. string.format(".Fruit(%d)", index)
        local fillType = g_fruitTypeManager:getFillTypeByFruitTypeIndex(fruitType.index)

        self.xmlFileDynamic:setInt(fruitKey .. "#id", fruitType.index)
        self.xmlFileDynamic:setString(fruitKey .. "#name", fillType.title)

        local periods = fruitType.growthDataSeasonal.periods

        for i = 1, 12 do
            local period = periods[i];
            local bePlanted = false
            local beHarvested = false

            if period then
                bePlanted = period.plantingAllowed
                beHarvested = period.isHarvestable
            end

            local periodicityKey = fruitKey .. string.format(".Periodicities.Periodicity(%d)", i)
            local name = g_i18n:formatPeriod(i, false)

            self.xmlFileDynamic:setInt(periodicityKey .. "#id", i)
            self.xmlFileDynamic:setString(periodicityKey .. "#name", name)

            self.xmlFileDynamic:setBool(periodicityKey .. "#bePlanted", bePlanted)
            self.xmlFileDynamic:setBool(periodicityKey .. "#beHarvested", beHarvested)
        end
    end

    return players
end

function Livemap:saveVehicles()
    local vehicles = {}
    local saveIndex = -1

    for vehicleId, vehicle in pairs(g_currentMission.vehicleSystem.vehicles) do
        if not vehicle.isDeleted and vehicle.trainSystem == nil or vehicle.typeName == "locomotive" then
            local mapHotspotType = vehicle.mapHotspotType

            if vehicle.typeName == "locomotive" then
                mapHotspotType = VehicleHotspot.TYPE.TRAIN
            end

            if mapHotspotType == VehicleHotspot.TYPE.TRAIN or vehicle.mapHotspot ~= nil and Utils:includes(Utils.excludeVehicleHotSpotTypes, mapHotspotType) == false then
                saveIndex = saveIndex + 1
                local key = ("Server.Vehicles.Vehicle(%d)"):format(saveIndex)
                local name = vehicle:getName()

                self.xmlFileDynamic:setInt(key .. "#id", vehicle.id)
                self.xmlFileDynamic:setString(key .. "#name", name)
                self.xmlFileDynamic:setString(key .. "#hotSpotType", Utils:vehicleHotSpotIndexToType(mapHotspotType))

                local ownerConnection = vehicle:getOwnerConnection()
                if ownerConnection ~= nil then
                    local user = g_currentMission.userManager:getUserByConnection(ownerConnection)

                    if user ~= nil and user:getNickname() ~= nil then
                        self.xmlFileDynamic:setString(key .. "#controller", user:getNickname())
                    end
                end

                local x, y, z = getWorldTranslation(vehicle.rootNode)
                local dx, _, dz = localDirectionToWorld(vehicle.rootNode, 0, 0, 1)
                local rotation = MathUtil.getYRotationFromDirection(dx, dz) + math.pi

                if x ~= nil and y ~= nil and z ~= nil then
                    self.xmlFileDynamic:setFloat(key .. "#x", x)
                    self.xmlFileDynamic:setFloat(key .. "#z", z)
                    self.xmlFileDynamic:setFloat(key .. "#r", rotation)
                end
            end
            --end
        end
    end

    return vehicles
end

function Livemap:savePlayers()
    local players = {}
    local serverUserId = 1
    local saveIndex = -1

    for _, user in pairs(g_currentMission.userManager:getUsers()) do
        if user:getId() ~= serverUserId then
            saveIndex = saveIndex + 1
            local key = ("Server.Players.Player(%d)"):format(saveIndex)

            self.xmlFileDynamic:setInt(key .. "#id", user:getId())

            local playtime = (g_currentMission.time - user:getConnectedTime()) / 60000
            self.xmlFileDynamic:setInt(key .. "#uptime", playtime)

            self.xmlFileDynamic:setString(key .. "#name", user:getNickname())

            local connection = user:getConnection()

            if connection ~= nil then
                local player = g_currentMission.connectionsToPlayer[connection]

                if player ~= nil and player.isControlled and player.rootNode ~= nil and player.rootNode ~= 0 then
                    local positionalInterpolator = player.positionalInterpolator
                    local x = positionalInterpolator.positionInterpolator.positionX
                    local z = positionalInterpolator.positionInterpolator.positionZ
                    local rotation = Utils:invertRotation(positionalInterpolator.yawInterpolator.value)

                    if x ~= nil and z ~= nil and rotation ~= nil then
                        self.xmlFileDynamic:setFloat(key .. "#x", x)
                        self.xmlFileDynamic:setFloat(key .. "#z", z)
                        self.xmlFileDynamic:setFloat(key .. "#r", rotation)
                    end
                end
            end
        end
    end

    return players
end

function Livemap:saveWeatherDaily()
    local weathers = {}

    for i = 1, 7 do
        local forecastInfo = self.forecast:getDailyForecast(i)

        local forecastType = Utils:weatherIndexToType(forecastInfo.forecastType)

        local period = self.environment:getPeriodFromDay(forecastInfo.day)
        local dayInPeriod = self.environment:getDayInPeriodFromDay(forecastInfo.day)

        local label = g_i18n:formatDayInPeriod(dayInPeriod, period, false)
        local highTemperature = Utils:formatTemperature(forecastInfo.highTemperature)
        local lowTemperature = Utils:formatTemperature(forecastInfo.lowTemperature)
        local windRotation = Utils:windDirectionToRadian(forecastInfo.windDirection)
        local windVelocity = Utils:mpsToBeaufort(forecastInfo.windSpeed)

        table.insert(weathers,
            {
                id = i,
                label = label,
                type = forecastType,
                highTemperature = highTemperature,
                lowTemperature =
                    lowTemperature,
                windRotation = windRotation,
                windVelocity = windVelocity
            })
    end

    self:WriteWeathersToXml(weathers, "Server.Weathers.Daily")
end

function Livemap:saveWeatherHourly()
    local weathers = {}

    local currentForecastInfo = self.forecast:getCurrentWeather()

    local currentWeatherType = Utils:weatherIndexToType(currentForecastInfo.forecastType)

    local currentTemp = Utils:formatTemperature(currentForecastInfo.temperature)

    local period = self.environment.currentPeriod
    local dayInPeriod = self.environment.currentDayInPeriod
    local label = g_i18n:formatDayInPeriod(dayInPeriod, period, false)

    local windRotation = Utils:windDirectionToRadian(currentForecastInfo.windDirection)
    local windVelocity = Utils:mpsToBeaufort(currentForecastInfo.windSpeed)

    --[[ During the getCurrentWeather, if the game is paused and that is not save with a progress game the value will be
        windSpeed = 1, windDirection = 90, so we are getting the firsr hourly forecast to have a real value, that's bad
        but there is no choice ]]
    if g_currentMission.paused == true then
        local firstHourForecast = self.forecast:getHourlyForecast(0)
        if firstHourForecast ~= nil then
            windRotation = Utils:windDirectionToRadian(firstHourForecast.windDirection)
            windVelocity = Utils:mpsToBeaufort(firstHourForecast.windSpeed)
        end
    end

    table.insert(weathers,
        {
            id = 0,
            label = label,
            type = currentWeatherType,
            temperature = currentTemp,
            windRotation = windRotation,
            windDirection = windDirection,
            windVelocity = windVelocity
        })

    -- 12 next hours
    for i = 1, 24 do
        if Utils:modulo(i, 2) ~= 0 then
            local forecastInfo = self.forecast:getHourlyForecast(i - 1)
            if forecastInfo ~= nil then
                local time = forecastInfo.time

                local weatherType = Utils:weatherIndexToType(forecastInfo.forecastType)

                local timeHours = math.floor(time / 3600000)
                local label = string.format("%02d:00", timeHours)

                local windRotation = Utils:windDirectionToRadian(forecastInfo.windDirection)

                local temperature = Utils:formatTemperature(forecastInfo.temperature)
                local windVelocity = Utils:mpsToBeaufort(forecastInfo.windSpeed)

                table.insert(weathers,
                    {
                        id = i,
                        label = label,
                        type = weatherType,
                        temperature = temperature,
                        windRotation = windRotation,
                        windVelocity =
                            windVelocity
                    })
            end
        end
    end

    self:WriteWeathersToXml(weathers, "Server.Weathers.Hourly")
end

function Livemap:saveTwister()
    if self.twister.isSpawned ~= true then
        return
    end

    local x, y, z = getWorldTranslation(self.twister.rootNode)

    local key = "Server.Twister";
    self.xmlFileDynamic:setInt(key .. "#id", self.twister.id)
    self.xmlFileDynamic:setInt(key .. "#speed", self.twister.metersPerHour)
    self.xmlFileDynamic:setFloat(key .. "#x", x)
    self.xmlFileDynamic:setFloat(key .. "#z", z)
end

function Livemap:WriteWeathersToXml(weathers, weatherKey)
    for weatherId, weatherType in pairs(weathers) do
        local key = string.format("%s(%d)", weatherKey, weatherId - 1)

        self.xmlFileDynamic:setInt(key .. "#id", weatherType.id)
        self.xmlFileDynamic:setString(key .. "#label", weatherType.label)
        self.xmlFileDynamic:setString(key .. "#type", weatherType.type)

        if weatherType.temperature ~= nil then
            self.xmlFileDynamic:setFloat(key .. "#temperature", weatherType.temperature)
        end

        if weatherType.lowTemperature ~= nil then
            self.xmlFileDynamic:setFloat(key .. "#lowTemperature", weatherType.lowTemperature)
        end

        if weatherType.highTemperature ~= nil then
            self.xmlFileDynamic:setFloat(key .. "#highTemperature", weatherType.highTemperature)
        end

        if weatherType.windRotation ~= nil then
            self.xmlFileDynamic:setFloat(key .. "#windRotation", weatherType.windRotation)
        end

        if weatherType.windVelocity ~= nil then
            self.xmlFileDynamic:setInt(key .. "#windVelocity", weatherType.windVelocity)
        end
    end
end

function Livemap:updateHotSpots()
    self.hotSpotsCache = self.hotSpotsCache or {}
    local hasChanges = false

    for _, placeable in pairs(self.placeableSystem.placeables) do
        if placeable ~= nil and
            placeable.spec_hotspots ~= nil and
            placeable.spec_hotspots.mapHotspots ~= nil and
            Utils:includes(Utils.excludePlaceableTypes, placeable.typeName) == false then
            local hotsSpots = placeable.spec_hotspots.mapHotspots

            if hotsSpots ~= nil then
                for id, hotSpot in pairs(hotsSpots) do
                    local cacheKey = (string.format("%i|%i", placeable.storeItem.id, id))

                    local name = placeable:getName()
                    local cacheEntry = self.hotSpotsCache[cacheKey]
                    local x = hotSpot.worldX
                    local z = hotSpot.worldZ
                    local typeIndex = hotSpot.placeableType
                    local typeName = Utils:placeableIndexToType(typeIndex)

                    if not cacheEntry or
                        cacheEntry.name ~= name or
                        cacheEntry.typeName ~= typeName or
                        cacheEntry.x ~= x or
                        cacheEntry.z ~= z or
                        cacheEntry.typeIndex ~= typeIndex then
                        hasChanges = true

                        self.hotSpotsCache[cacheKey] = {
                            name = name,
                            typeName = typeName,
                            x = x,
                            z = z,
                            typeIndex = typeIndex
                        }
                    end
                end
            end
        end
    end

    return hasChanges
end

function Livemap:writeHotSpots()
    local saveIndex = 0

    for _, placeable in pairs(self.hotSpotsCache) do
        saveIndex = saveIndex + 1
        local key = ("Server.HotSpots.HotSpot(%d)"):format(saveIndex)

        self.xmlFileStatic:setString(key .. "#name", placeable.name)
        self.xmlFileStatic:setString(key .. "#type", placeable.typeName)
        self.xmlFileStatic:setInt(key .. "#id", saveIndex)
        self.xmlFileStatic:setFloat(key .. "#x", placeable.x)
        self.xmlFileStatic:setFloat(key .. "#z", placeable.z)
    end
end

function Livemap:updateFields()
    self.fieldCache = self.fieldCache or {}
    local hasChanges = false

    for fieldId, field in pairs(g_fieldManager.fields) do
        if field.farmland ~= nil then
            local farmlandId = field.farmland.id
            local cacheKey = tostring(fieldId)

            local x, z = field:getCenterOfFieldWorldPosition()
            local fruitTypeIndexPos = FSDensityMapUtil.getFruitTypeIndexAtWorldPos(x, z)
            local fruitTypeDesc = nil

            if fruitTypeIndexPos ~= nil then
                fruitTypeDesc = g_fruitTypeManager:getFruitTypeByIndex(fruitTypeIndexPos)
            end

            local fruitColor = "rgba(0, 0, 0, 255)"
            if fruitTypeDesc then
                local defaultMapColor = fruitTypeDesc.defaultMapColor
                if defaultMapColor ~= nil then
                    if defaultMapColor.r ~= nil and defaultMapColor.g ~= nil and defaultMapColor.b ~= nil then
                        local r = Utils:giantsToRgb(defaultMapColor.r)
                        local g = Utils:giantsToRgb(defaultMapColor.g)
                        local b = Utils:giantsToRgb(defaultMapColor.b)
                        fruitColor = ("rgba(%d, %d, %d, 255)"):format(r, g, b)
                    end
                end
            end

            local owner = Utils:getFarmlandOwner(field.farmland)
            -- local finalFruitType = fruitTypeDesc and fruitTypeDesc.name or "unknown"
            local finalFruitType = fruitTypeDesc and fruitTypeDesc.fillType.title or "unknown"

            local cacheEntry = self.fieldCache[cacheKey]
            if not cacheEntry or
                cacheEntry.price ~= field.farmland.price or
                cacheEntry.owner ~= owner or
                cacheEntry.fruitName ~= finalFruitType or
                cacheEntry.farmlandPrice ~= field.farmland.price or
                cacheEntry.farmlandArea ~= field.farmland.areaInHa or
                cacheEntry.farmland ~= farmlandId or
                cacheEntry.fruitColor ~= fruitColor then
                hasChanges = true

                self.fieldCache[cacheKey] = {
                    price = field.farmland.price,
                    owner = owner,
                    fruitName = finalFruitType,
                    farmland = farmlandId,
                    farmlandPrice = field.farmland.price,
                    farmlandArea = field.farmland.areaInHa,
                    fruitColor = fruitColor,
                    polygons = self:getFieldPolygons(field, fieldId == 1)
                }
            end
        end
    end

    return hasChanges
end

function Livemap:writeFields()
    for cacheKey, cacheEntry in pairs(self.fieldCache) do
        local fieldId = tonumber(cacheKey)
        local fieldKey = ("Server.Fields.Field(%d)"):format(fieldId - 1)

        self.xmlFileStatic:setInt(fieldKey .. "#id", fieldId)
        self.xmlFileStatic:setString(fieldKey .. "#price", "todo")
        self.xmlFileStatic:setString(fieldKey .. "#fruitColor", cacheEntry.fruitColor)
        self.xmlFileStatic:setString(fieldKey .. "#fruitName", cacheEntry.fruitName)
        self.xmlFileStatic:setInt(fieldKey .. "#farmland", cacheEntry.farmland)
        self.xmlFileStatic:setInt(fieldKey .. "#farmlandPrice", cacheEntry.farmlandPrice)
        self.xmlFileStatic:setFloat(fieldKey .. "#farmlandArea", cacheEntry.farmlandArea)

        if cacheEntry.owner ~= nil then
            self.xmlFileStatic:setString(fieldKey .. "#farmlandOwner", cacheEntry.owner)
        end

        self:writePolygonToXml(cacheEntry.polygons, fieldKey)
    end
end

function Livemap:writePolygonToXml(polygons, parentKey)
    for polygonId, polygon in ipairs(polygons) do
        local polygonKey = parentKey .. ".P(" .. (polygonId - 1) .. ")"

        local pointParts = {}
        for i = 1, #polygon do
            local point = polygon[i]
            pointParts[i] = string.format("%.1f,%.1f", point.x, point.z)
        end

        local pointsString = table.concat(pointParts, " ")
        self.xmlFileStatic:setString(polygonKey .. "#pts", pointsString)
    end
end

function Livemap:getFieldPolygons(field, a)
    local polygons = {}
    local points   = {};

    for pointId, point in ipairs(field.densityMapPolygon.pointsZ) do
        local pointX = field.densityMapPolygon.pointsX[pointId];
        table.insert(points, { x = pointX, z = point, i = pointId, d = 0 })
    end

    table.insert(polygons, points)

    return polygons
end

function Livemap:getSeasonPercentageProgress()
    local daysPerPeriod = self.environment.daysPerPeriod
    local currentDay = self.environment.currentDayInPeriod
    local currentPeriod = self.environment.currentPeriod
    local totalPeriods = 12
    local progressInPeriod = ((currentDay - 1) + 0.5) / daysPerPeriod

    local progress = (currentPeriod - 1 + progressInPeriod) / totalPeriods
    return progress * 100
end

FSBaseMission.onStartMission = Utils.appendedFunction(FSBaseMission.onStartMission, onStart)
