vsDebug("VehicleShopStorage Class")

VehicleShopStorage = {}
VehicleShopStorage.modName = g_currentModName
VehicleShopStorage.modDirectory = g_currentModDirectory

local VehicleShopStorage_mt = Class(VehicleShopStorage, Event)

InitEventClass(VehicleShopStorage, "VehicleShopStorage")

function VehicleShopStorage.new(mission, i18n, modDirectory, modName)
  vsDebug("VehicleShopStorage-New")
  local self = setmetatable({}, VehicleShopStorage_mt)
  self.mission                = mission
  self.i18n                   = i18n
  self.modDirectory           = modDirectory
  self.modName                = modName
  self.curVeh                 = nil
  self.currentVehicle         = nil
  self.storageButton          = nil
  self.viewStorageButton      = nil
  self.vehicleSpawnDebug      = false
  self.vehicleSpawnDebugTable = nil

  addConsoleCommand("gsStoreVehicle", "Stores current vehicle to players garage.", "consoleCommandStoreVehicle", self)

	return self
end

-- Run on map load
function VehicleShopStorage:loadMap()
  vsDebug(" Info: VehicleShopStorage-loadMap")

  g_workshopStorageScreen = WorkshopStorageScreen.register()

end

function VehicleShopStorage:update()
  if self.vehicleSpawnDebug then


    -- Function to draw a line between two points
    function drawLineBetweenPoints(startX, startY, startZ, endX, endY, endZ, color)
        local r, g, b = unpack(color) -- Unpack RGB values
        drawDebugLine(startX, startY, startZ, r, g, b, endX, endY, endZ, r, g, b, true)
    end

    -- Function to handle a single shop location
    function drawShopLocationLine(shopLocation)
      if shopLocation.lineStart ~= nil and shopLocation.lineEnd ~= nil then  

        -- vsDebug("shopLocation")
        -- vsDebug(shopLocation)

        -- Extract starting point
        local startX = shopLocation.lineStart.x
        local startY = shopLocation.lineStart.y + 1
        local startZ = shopLocation.lineStart.z

        local endX = shopLocation.lineEnd.x
        local endY = shopLocation.lineEnd.y + 1
        local endZ = shopLocation.lineEnd.z

        if shopLocation.nodeStart ~= nil and shopLocation.nodeEnd ~= nil then
          DebugUtil.drawDebugNode(shopLocation.nodeStart, "spawnStart")
          DebugUtil.drawDebugNode(shopLocation.nodeEnd, "spawnEnd")
        end

        -- drawDebugPoint(startX, startY + 1, startZ, 1, 1, 1, 1)
        -- drawDebugPoint(startX, startY + 1, startZ, 1, 1, 1, 1)

        -- Define a color (red in this case)
        local color = {0, 0, 1} -- RGB values (1 = full, 0 = none)

        -- Draw a line between the start and end points
        drawLineBetweenPoints(startX, startY, startZ, endX, endY, endZ, color)

      end
    end

    local storeSpawnLocations = nil

    if g_workshopStorageScreen.currentShopLocation ~= nil then
      storeSpawnLocations = g_workshopStorageScreen.currentShopLocation
    else
      if #g_currentMission.loadSpawnPlaces > 0 then
        storeSpawnLocations = g_currentMission.loadSpawnPlaces
      else
        storeSpawnLocations = g_currentMission.storeSpawnPlaces
      end
    end

    -- Draw lines for each shop location
    if storeSpawnLocations ~= nil then
        -- Only show this once
        if storeSpawnLocations ~= self.vehicleSpawnDebugTable then
          vsDebug("storeSpawnLocations changed")
          vsDebug(storeSpawnLocations)
          self.vehicleSpawnDebugTable = storeSpawnLocations
        end
        -- Cycle though the spawn locations to draw them
        for i, shopLocation in ipairs(storeSpawnLocations) do
            drawShopLocationLine(shopLocation)
        end
    end

  end
end

function VehicleShopStorage:consoleCommandStoreVehicle()
  vsDebug("Info: VehicleShopStorage:consoleCommandStoreVehicle")
  -- Get vehicle data for current user
  self.curVeh = g_currentMission.hud.controlledVehicle
  if self.curVeh ~= nil then 
    self.currentVehicle = self.curVeh
  elseif self.hudUpdater.object ~= nil and self.hudUpdater.object.isActive ~= nil and self.hudUpdater.isVehicle then
    self.currentVehicle = self.hudUpdater.object
  else
    self.currentVehicle = nil
  end
  -- Do stuff with vehicle
  if self.currentVehicle ~= nil then
    -- Check if is server or host.  If host, let the user do what is needed.
    if self.isServer then
      -- User is server or host
      vsDebug('  Info:VehicleShopStorage:consoleCommandStoreVehicle: user is server.')
      -- Since we are local, go ahead store vehicle
      VehicleStorageEvent.sendEvent(self.currentVehicle)
      vsDebug("Selected Vehicle was sent to storage.")
    else
      -- Make sure user is a client
      if self.isClient then
        -- User is client.  Make sure they are admin then broadcast to server
        if g_currentMission.isMasterUser then
          vsDebug('  Info:VehicleShopStorage:consoleCommandStoreVehicle: user is client and admin.')
          -- Send event data to server
          VehicleStorageEvent.sendEvent(self.currentVehicle)
          vsDebug("Selected Vehicle was sent to storage.")
        else
          vsDebug('  Info:VehicleShopStorage:consoleCommandStoreVehicle: user is client and not admin.')
          print('Info: Must be logged in as admin to run that command.')
        end
      end
    end
  else
    print("Info: No Vehicle Found.  You must be next to vehicle or in it to use this command.")
  end
end


-- Store the vehicle.  Save xml to outbox and remove from game.
function VehicleShopStorage:storeVehicle(vehicle)
  vsDebug("VehicleShopStorage-storeVehicle")

  if vehicle ~= nil then

    -- Check to see if we are on a server or local.  SP should be 0 and MP should have a farm number
    local vehicleOwnerFarmId = 0
    if g_currentMission.missionDynamicInfo.isMultiplayer then
      vehicleOwnerFarmId = vehicle.ownerFarmId
    end

    local modName         = g_currentModName or "FS25_VehicleShopStorage"
    local modProfilePath  = getUserProfileAppPath()

    local modSettingsFolderPath = modProfilePath  .. "modSettings/" .. modName
    local vehiclesFolderPath = modSettingsFolderPath  .. "/vehicles"

    -- Folder path where the vehicles will store for current farm vehicle
    local vehicleStorageDir = vehiclesFolderPath .. "/" .. vehicleOwnerFarmId .. "/"
    -- Create folder is not already created
    if ( not fileExists(vehicleStorageDir) ) then createFolder(vehicleStorageDir) end
    local vehicleFileName = vehicleStorageDir .. "storeVehicle-" .. vehicleOwnerFarmId .. "-" .. vehicle.uniqueId .. ".xml"
    local vehicleXMLFile = XMLFile.create("vehicleXMLFile", vehicleFileName, "vehicles", Vehicle.xmlSchemaSavegame)
    
    if vehicleXMLFile ~= nil then
      local key = string.format("vehicles.vehicle(%d)", 0)
      vehicle.currentSavegameId = 1
      local vehicle_getSellPrice = "0"
      if vehicle.getSellPrice ~= nil then vehicle_getSellPrice = vehicle:getSellPrice() end 
      local modName = vehicle.customEnvironment
      if modName ~= nil then
        vehicleXMLFile:setValue(key .. "#modName", modName)
      end
      -- More Details For Website Display
      vehicle:saveToXMLFile(vehicleXMLFile, key, {})
      vehicleXMLFile:setValue(key .. "#filename", HTMLUtil.encodeToHTML(NetworkUtil.convertToNetworkFilename(vehicle.configFileName)))
      vehicleXMLFile:save()
      vehicleXMLFile:delete()
      -- Remove vehicle from game
      vehicle:delete()
      -- Charge the farm for the transfer fee
      local farm = g_farmManager:getFarmById(vehicle:getOwnerFarmId())
      if farm ~= nil then
      local storageCost = -100
      local moneyType = MoneyType.VEHICLE_RUNNING_COSTS
      g_currentMission:addMoneyChange(storageCost, vehicleOwnerFarmId, moneyType, true)
        vsDebug("Charging the farm for storing vehicle.")
        farm:changeBalance(storageCost, moneyType)
      end
    end
  else
    vsDebug("Vehicle Data Missing")
  end
end

-- Load vehicle from xml file
function VehicleShopStorage:loadVehicle(xmlFileVehicle, farmId, spawnLocationData, userId)
  vsDebug("VehicleShopStorage-loadVehicle")

  -- If farmId not set then set it to 0
  if farmId == nil then
    farmId = 0
  end

  -- Update the spawn location data
  if spawnLocationData ~= nil then
    -- Put the spawn location back into a table
    local updatedSpawnLocationData = {}
    table.insert(updatedSpawnLocationData,spawnLocationData)
    g_workshopStorageScreen:updateShopLocation(updatedSpawnLocationData)
  end

  vsDebug("xmlFileVehicle")
  vsDebug(xmlFileVehicle)

  -- check if filename is remote or local called and correct the path if needed
  local loadXmlFileVehicle = xmlFileVehicle
  if not fileExists(loadXmlFileVehicle) then
    local modName         = g_currentModName or "FS25_VehicleShopStorage"
    local modProfilePath  = getUserProfileAppPath()

    local modSettingsFolderPath = modProfilePath  .. "modSettings/" .. modName
    local vehiclesFolderPath = modSettingsFolderPath  .. "/vehicles"

    -- Folder path where the vehicles will store for current farm vehicle
    local vehicleStorageDir = vehiclesFolderPath .. "/" .. farmId .. "/"
    loadXmlFileVehicle = vehicleStorageDir .. xmlFileVehicle
  end

  vsDebug("loadXmlFileVehicle")
  vsDebug(loadXmlFileVehicle)

  if fileExists(loadXmlFileVehicle) then
    local xmlFile = XMLFile.load("VehiclesXML", loadXmlFileVehicle, Vehicle.xmlSchemaSavegame)
    local i = 1
    local key = string.format("vehicles.vehicle(%d)", i - 1)

    local vehicleUniqueId = xmlFile:getString(key .. "#uniqueId")
    local storeFilename = xmlFile:getString(key .. "#filename")

    self.vehiclesToSpawnLoading = true
    local args = {
      xmlFilename = loadXmlFileVehicle,
      key = key,
      xmlFile = xmlFile,
      vehicleUniqueId = vehicleUniqueId,
      storeFilename = storeFilename,
      farmId = farmId,
      userId = userId,
    }

    ShowMessageDialogEvent.sendEvent(userId,g_i18n:getText("info_spawningVehicle"))

    local vehicleSystem = g_currentMission.vehicleSystem

    -- Callback function for the vehicle load
    local function asyncCallbackFunction(_, vehicle, vehicleLoadState, arguments)
      
      -- Make sure we have a vehicle to reset
      if vehicle ~= nil and arguments.vehicleUniqueId ~= nil then
        local vehicleToReset = g_currentMission.vehicleSystem:getVehicleByUniqueId(arguments.vehicleUniqueId)
        if vehicleToReset ~= nil then
          -- Send the vehicle to spawn area
          g_client:getServerConnection():sendEvent(ResetVehicleEvent.new(vehicleToReset))
        end
      end

      -- Hide the loading thingy
      HideMessageDialogEvent.sendEvent(arguments.userId)

      if vehicleLoadState == VehicleLoadingState.OK then
        print("Vehicle Spawned From Storage.  Remove from storage files.")

        -- Delete the xml from the storage folder
        if fileExists(arguments.xmlFilename) then
          deleteFile(arguments.xmlFilename)
        end
        VehicleReloadEvent.sendEvent(arguments.farmId)
        ShowDialogEvent.sendEvent(userId,g_i18n:getText("info_spawnSuccess"),DialogElement.TYPE_INFO)
        return true
      else
        printf("Warning: vehicle xml '%s', vehicle '%s' could not be loaded", arguments.xmlFilename, arguments.key)
        ShowDialogEvent.sendEvent(arguments.userId,g_i18n:getText("info_spawnError"),DialogElement.TYPE_WARNING)
        return false
      end
    end

    -- Load the vehicle from storage files
    g_asyncTaskManager:addTask(function()
      vehicleSystem:loadFromXMLFile(xmlFile, asyncCallbackFunction, nil, args, false, false)
    end)

  else
    printf("Warning: unable to load selected vehicle.");
    ShowDialogEvent.sendEvent(userId,g_i18n:getText("info_spawnError"),DialogElement.TYPE_WARNING)
  end
end

-- Add store vehicle button to shop menu
function VehicleShopStorage:setVehicle(vehicle, second)
  vsDebug("VehicleShopStorage-setVehicle")

  if self.owner ~= nil and self.owner.owningPlaceable ~= nil then
    g_vehicleShopStorage:buildShopSpawnLocation(self.owner.owningPlaceable)
  end

  -- Add button to view stored vehicles
  if self.viewStorageButton == nil then
    local ViewStorageElement = self.sellButton:clone()
    ViewStorageElement:setText(g_i18n:getText("button_viewStorage"))
    ViewStorageElement:setInputAction("MENU_PAGE_NEXT")
    ViewStorageElement:setVisible(true)
    ViewStorageElement:setDisabled(false)
    ViewStorageElement.onClickCallback = function ()
      VehicleShopStorage:vehicleStorage_openGui()
    end
    self.sellButton.parent:addElement(ViewStorageElement)
    self.viewStorageButton = ViewStorageElement

    -- Check if user has permission to store vehicle
    if g_currentMission.missionDynamicInfo.isMultiplayer then
      if not g_currentMission:getHasPlayerPermission("farmManager") then
        ViewStorageElement:setDisabled(true)
      end
    end
  end


  if vehicle ~= nil then 
    if vehicle.propertyState == VehiclePropertyState.OWNED then

      -- Get vehicle id
      if vehicle.id ~= self.previousVehicle then

        vsDebug("Vehicle Owned")
      
        -- Need to remove the old button when the vehicle changes to create a new one.
        if self.storageButton ~= nil then
          self.storageButton:unlinkElement()
          self.storageButton:delete()
          self.storageButton = nil
        end

        -- Add button to store selected vehicle
        if self.storageButton == nil then
          local storageCost = 100
          local StorageElement = self.sellButton:clone()
          StorageElement:setText(string.format("%s (%s)", g_i18n:getText("button_storeVehicle"),g_i18n:formatMoney(storageCost, 0, true, true)))
          StorageElement:setInputAction("TOGGLE_STORE")
          StorageElement:setVisible(vehicle ~= nil)
          StorageElement:setDisabled(false)
          StorageElement.onClickCallback = function ()
            VehicleShopStorage:storeVehicleConfirm(vehicle)
          end
          self.sellButton.parent:addElement(StorageElement)
          self.storageButton = StorageElement

          self.previousVehicle = vehicle.id

          -- Check if user has permission to store vehicle
          if g_currentMission.missionDynamicInfo.isMultiplayer then
            if not g_currentMission:getHasPlayerPermission("farmManager") then
              StorageElement:setDisabled(true)
            end
          end

        end
      end
    else
      vsDebug("Vehicle Not Owned")
      if self.storageButton ~= nil then
        self.storageButton:setText(string.format("%s", g_i18n:getText("button_storeVehicle")))
        self.storageButton:setDisabled(true)
      end
    end
  else
    vsDebug("No Vehicles Found")
    if self.storageButton ~= nil then
      self.storageButton:setText(string.format("%s", g_i18n:getText("button_storeVehicle")))
      self.storageButton:setDisabled(true)
    end
  end

end

-- Update spawn location for vehicle shop
function VehicleShopStorage:buildShopSpawnLocation(owningPlaceable)

  if owningPlaceable ~= nil and owningPlaceable.spec_workshop ~= nil and owningPlaceable.spec_workshop.sellingPoint ~= nil and owningPlaceable.spec_workshop.sellingPoint.sellTriggerNode ~= nil then

    -- Get the owning placeable's node and the workshop sell vehicle trigger node
    local owningPlaceableNode = owningPlaceable.rootNode
    local triggerNode = owningPlaceable.spec_workshop.sellingPoint.sellTriggerNode

    -- Get the local position of the trigger node relative to the owning placeable
    local localX, localY, localZ = localToLocal(triggerNode, owningPlaceableNode, 0, 0, 0)

    -- Get the owning placeable's world rotation (in radians)
    local placeableRotX, placeableRotY, placeableRotZ = getRotation(owningPlaceableNode)

    -- Calculate the world position of the trigger node considering placeable rotation
    local centerX, centerY, centerZ = localToWorld(owningPlaceableNode, localX, localY, localZ)

    -- Get the world rotation of the trigger node (add placeable rotation to local rotation)
    local triggerRotX, triggerRotY, triggerRotZ = getRotation(triggerNode)
    local centerRX = triggerRotX + placeableRotX
    local centerRY = triggerRotY + placeableRotY
    local centerRZ = triggerRotZ + placeableRotZ

    -- vsDebug({"Location", centerX, centerY, centerZ})
    -- vsDebug({"Rotation", centerRX, centerRY, centerRZ})

    -- Create temp transforms to get cords for vehicle spawn area
    local spawnAdjust = createTransformGroup("spawnAdjust")
    local spawnStart = createTransformGroup("spawnStart")
    local spawnEnd = createTransformGroup("spawnEnd")

    link(owningPlaceableNode, spawnAdjust)
    link(spawnAdjust, spawnStart)
    link(spawnStart, spawnEnd)

    -- Check to see if we have a custom preset for the offsets based on the building
    -- vsDebug("owningPlaceable")
    -- vsDebug(owningPlaceable)

    -- Default offset to hopefully spawn vehicles in good places
    local spawnOffset = {
      x = 8, -- distance from center -left to right
      z = 4, -- distance from center -back to front
      width = 16
    }

    -- Set the offsets for default and map workshops
    if owningPlaceable.storeItem.xmlFilenameLower == "data/placeables/brandless/workshop/workshop.xml" then
      -- Set the offsets for the default in game workshop
      spawnOffset = {
        x = -8, -- distance from center -left to right
        z = 14, -- distance from center -back to front
        width = 16
      }
    elseif owningPlaceable.storeItem.xmlFilenameLower == "data/placeables/mapus/sellingpoints/sellingstationvehicles/sellingstationvehicles.xml" then
      -- Set the offsets for the default in game us map workshop
      spawnOffset = {
        x = 8, -- distance from center -left to right
        z = 4, -- distance from center -back to front
        width = 16
      }
    elseif owningPlaceable.storeItem.xmlFilenameLower == "data/placeables/mapas/sellingpoints/sellingstationvehicles/sellingstationvehicles.xml" then
      -- Set the offsets for the default in game as map workshop
      spawnOffset = {
        x = 2, -- distance from center -left to right
        z = 24, -- distance from center -back to front
        width = 16
      }
    elseif owningPlaceable.storeItem.xmlFilenameLower == "data/placeables/mapeu/sellingstationvehicles/sellingstationvehicles.xml" then
      -- Set the offsets for the default in game eu map workshop
      spawnOffset = {
        x = 12, -- distance from center -left to right
        z = 4, -- distance from center -back to front
        width = 10
      }
    end

    setWorldTranslation(spawnStart, centerX, centerY, centerZ)
    setWorldRotation(spawnStart, centerRX, centerRY, centerRZ)
    setTranslation(spawnStart, spawnOffset.x, 0, spawnOffset.z) -- sets the offset for the spawn area
    setTranslation(spawnEnd, spawnOffset.width, 0, 0) -- sets the width of the spawn area

    -- vsDebug("spawnStart")
    -- vsDebug(spawnStart)

    -- Load the placement cords for shop stuffs
    local placeNode = PlacementUtil.loadPlaceFromNode(spawnStart)

    -- vsDebug("placeNode")
    -- vsDebug(placeNode)

    local sx, sy, sz = getWorldRotation(spawnStart)
    local ex, ey, ez = getWorldRotation(spawnEnd)

    placeNode.nodeStart = spawnStart
    placeNode.nodeEnd = spawnEnd
    placeNode.lineStart = {x = sx, y = sy, z = sz}
    placeNode.lineEnd = {x = ex, y = ey, z = ez}

    -- Add all spawns to the spawn data table
    local newSpawnsData = {placeNode}

    -- send the new vehicle spawn locations to the store spawn places
    g_workshopStorageScreen:updateShopLocation(newSpawnsData)

  end

end

-- Confirm with player to store vehicle
function VehicleShopStorage:storeVehicleConfirm(vehicle)
	YesNoDialog.show(function(yes)
    if yes then
      if g_currentMission.missionDynamicInfo.isMultiplayer then
        VehicleStorageEvent.sendEvent(vehicle)
      else
        VehicleShopStorage:storeVehicle(vehicle)
      end
    end
	end, nil, g_i18n:getText("button_storeVehicleConfirm"))
end

-- Load the gui
function VehicleShopStorage:vehicleStorage_openGui()
  vsDebug(" VehicleShopStorage-vehicleStorage_openGui")

  -- Load the gui
  g_gui:showGui("WorkshopStorageScreen")

  -- Clear current vehicles if any from list incase a new farm requesting
  g_workshopStorageScreen:clearVehicleList()
  g_workshopStorageScreen.list:reloadData()

  -- If mp use the farm id, if sp use 0
  local farmId = 0;
  if g_currentMission.missionDynamicInfo.isMultiplayer then
    farmId = g_localPlayer:getFarmId()
  end

  -- Load the Vehicles
  VehicleReloadEvent.sendEvent(farmId)
  
end

-- Get vehicles in storage and send them out to the network
function VehicleShopStorage:requestFarmVehicles(farmId)
  vsDebug(" VehicleShopStorage-requestFarmVehicles")

  VehicleReloadEvent.sendEvent(farmId)
end