--
-- Workshop Storage Screen
--

WorkshopStorageScreen = {}
WorkshopStorageScreen.modDirectory = g_currentModDirectory

local WorkshopStorageScreen_mt = Class(WorkshopStorageScreen, ScreenElement)

function WorkshopStorageScreen:new(mission, i18n, modDirectory, modName, subclass_mt)
    local self = WorkshopStorageScreen:superClass().new(nil, subclass_mt or WorkshopStorageScreen_mt)

    vsDebug("WorkshopStorageScreen:new")

    self.mission                = mission
    self.i18n                   = i18n
    self.modDirectory           = modDirectory
    self.modName                = modName
    self.setValueTimerFrequency = 60
    self.storedVehicles         = {}
    self.currentShopLocation    = nil

    local modName         = g_currentModName or "FS25_VehicleShopStorage"
    local modProfilePath  = getUserProfileAppPath()

    local modSettingsFolderPath = modProfilePath  .. "modSettings/" .. modName
    self.vehiclesFolderPath = modSettingsFolderPath  .. "/vehicles"

    self.requestedFarmId = 0

    return self
end

function WorkshopStorageScreen.register()
  vsDebug(" Info: WorkshopStorageScreen-register")
  local workshopStorageScreen = WorkshopStorageScreen:new(nil, g_i18n)
  g_gui:loadGui(WorkshopStorageScreen.modDirectory .. "xml/WorkshopStorageScreen.xml", "WorkshopStorageScreen", workshopStorageScreen)
  return workshopStorageScreen
end


function WorkshopStorageScreen:reloadVehicles(farmId)
  vsDebug("WorkshopStorageScreen:reloadVehicles")

  self.requestedFarmId = farmId

  -- Folder path where the vehicles will store for current farm vehicle
  local vehicleStorageDir = self.vehiclesFolderPath .. "/" .. self.requestedFarmId .. "/"

  getFiles(vehicleStorageDir, "loadVehicles", self)

end

function WorkshopStorageScreen:loadVehicles(xmlFileVehicle)
  vsDebug("WorkshopStorageScreen:loadVehicles")

  -- Check if vehicle is in the list, if not then add it.

  -- Folder path where the vehicles will store for current farm vehicle
  local vehicleStorageDir = self.vehiclesFolderPath .. "/" .. self.requestedFarmId .. "/"

  local loadXmlFileVehicle = vehicleStorageDir .. xmlFileVehicle

  vsDebug(loadXmlFileVehicle)

  if loadXmlFileVehicle ~= nil and fileExists(loadXmlFileVehicle) then

    local xmlFile = XMLFile.load("VehiclesXML", loadXmlFileVehicle, Vehicle.xmlSchemaSavegame)

    xmlFile:iterate("vehicles.vehicle", function(_, vehicleData)
      
      local uniqueId = xmlFile:getString(vehicleData .. "#uniqueId");
      local filename = xmlFile:getString(vehicleData .. "#filename");
      local age = xmlFile:getValue(vehicleData .. "#age");
      local price = xmlFile:getValue(vehicleData .. "#price");
      local operatingTime = xmlFile:getValue(vehicleData .. "#operatingTime");
      local damage = xmlFile:getValue(vehicleData .. ".wearable#damage");
      local paint = xmlFile:getFloat(vehicleData .. ".wearable.wearNode#amount");
      local loadFile = NetworkUtil.convertFromNetworkFilename(filename)
      local storeItem = g_storeManager:getItemByXMLFilename(loadFile)

      if storeItem ~= nil then

        local curVehicleData = {
          uniqueId = uniqueId,
          filename = filename,
          age = age,
          price = price,
          operatingTime = operatingTime,
          damage = damage,
          paint = paint,
          storeItem = storeItem,
          vehicleXmlFile = loadXmlFileVehicle
        }
      
        -- vsDebug("curVehicleData")
        -- vsDebug(curVehicleData)

        -- add the vehicle to the storage if not already
        local vehicleMatch = false
        if self.storedVehicles ~= nil then
          for _, storedVehicle in pairs(self.storedVehicles) do
            if storedVehicle.uniqueId == curVehicleData.uniqueId then
              -- vehicle already loaded
              vehicleMatch = true
            end
          end
        end 

        -- If no match then add vehicle to the list 
        if not vehicleMatch then 
          table.insert(self.storedVehicles, curVehicleData)
        end
      
      end

    end)

    -- Vehicles loaded
    if self.storedVehicles ~= nil then
      -- vsDebug("Stored Vehicles Table")
      -- vsDebug(self.storedVehicles)

      -- Set the vehicles
      for _, vehicle in pairs(self.storedVehicles) do
        -- Send to the vehicle list event
        VehicleListEvent.sendEvent(self.requestedFarmId,vehicle.uniqueId,vehicle.filename,vehicle.age,vehicle.price,vehicle.operatingTime,vehicle.damage,vehicle.paint,xmlFileVehicle)
      end

    end

    if g_client ~= nil then
      self.list:reloadData()
    end

  end

end

function WorkshopStorageScreen:loadVehicleRequest(data,farmId)
  vsDebug("WorkshopStorageScreen:loadVehicleRequest")
  -- Only reload for the player that requested this set of vehicles
  if g_localPlayer:getFarmId() == farmId then

    -- Adds vehicle to table for view in mp

    local curVehicleData = {
      uniqueId = data.uniqueId,
      filename = data.filename,
      age = data.age,
      price = data.price,
      operatingTime = data.operatingTime,
      damage = data.damage,
      paint = data.paint,
      storeItem = data.storeItem,
      vehicleXmlFile = data.vehicleXmlFile,
      farmId = farmId
    }

    vsDebug("curVehicleData")
    vsDebug(curVehicleData)

    -- add the vehicle to the storage if not already
    local vehicleMatch = false
    if self.storedVehicles ~= nil then
      for _, storedVehicle in pairs(self.storedVehicles) do
        if storedVehicle.uniqueId == curVehicleData.uniqueId then
          -- vehicle already loaded
          vehicleMatch = true
        end
      end
    end 

    -- If no match then add vehicle to the list 
    if not vehicleMatch then 
      table.insert(self.storedVehicles, curVehicleData)
    end

    self.list:reloadData()

  end
end

function WorkshopStorageScreen.getNumberOfItemsInSection(self, _, _)
  vsDebug("WorkshopStorageScreen:getNumberOfItemsInSection")

	return #self.storedVehicles
end


function WorkshopStorageScreen.setStatusBarValue(_, bar, value)
  vsDebug("WorkshopStorageScreen:setStatusBarValue")

	local newValue = (bar.lastStatusBarValue or -1) - value
	if math.abs(newValue) > 0.01 then
		local width = bar.parent.size[1] - bar.margin[1] * 2
		local start = bar.startSize == nil and 0 or bar.startSize[1] + bar.endSize[1]
		local length = width * math.min(value, 1)
		bar:setSize(math.max(start, length), nil)
		bar.lastStatusBarValue = value
	end
end

function WorkshopStorageScreen.setVehicles(self, vehicles)
  vsDebug("WorkshopStorageScreen:setVehicles")

	self.vehicles = vehicles
	self.list:reloadData()
	if #vehicles == 0 then
		self:setVehicle(nil)
	end
end

function WorkshopStorageScreen.updateVehicles(self, owner, vehicles)
  vsDebug("WorkshopStorageScreen:updateVehicles")

	self:setVehicles(vehicles)
end

function WorkshopStorageScreen.onListSelectionChanged(self, _, _, item)
  vsDebug("WorkshopStorageScreen:onListSelectionChanged")

	self:setVehicle(self.storedVehicles[item])
end

function WorkshopStorageScreen.setVehicle(self, vehicle)
  vsDebug("WorkshopStorageScreen:setVehicle")

	self.storeItem = nil
	self.canBeConfigured = false
	if vehicle == nil then
		self.spawnButton:setLocaKey("button_spawn")
		self.spawnButton:setDisabled(true)
	else
		self.vehicle = vehicle
		self.storeItem = vehicle.storeItem
		self.spawnButton:setLocaKey("button_spawn")
		self.spawnButton:setDisabled(false)
	end

	self.spawnButton:setVisible(true)
	self.dialogInfo:setVisible(vehicle == nil)
	self.buttonsBox:invalidateLayout()
end

function WorkshopStorageScreen.populateCellForItemInSection(self, _, _, row, vehList)
  vsDebug("WorkshopStorageScreen:populateCellForItemInSection")

	local vehicle = self.storedVehicles[row]

  local loadFile = NetworkUtil.convertFromNetworkFilename(vehicle.filename)

  -- Check to make sure vehicle can be loaded
  local storeItem = g_storeManager:getItemByXMLFilename(loadFile)
  if storeItem ~= nil then
    vehList:getAttribute("icon"):setImageFilename(self:getImageFilename(loadFile))
    
    local brand = g_brandManager:getBrandByIndex(vehicle.storeItem.brandIndex)
    local brandTitle = brand == nil and "" or brand.title .. " "
    vehList:getAttribute("name"):setText(brandTitle .. self:getName(loadFile))
    self:setVehicleDetails(vehicle, vehList)
  end
end

function WorkshopStorageScreen.setVehicleDetails(self, vehicle, vehList)
  vsDebug("WorkshopStorageScreen:setVehicleDetails")

	local operatingTime, age
	if vehicle.uniqueId ~= nil and vehicle.operatingTime ~= nil and vehicle.age ~= nil then
		operatingTime = vehicle.operatingTime
		age = vehicle.age
    local price = vehicle.price
    if price ~= nil then
      vehList:getAttribute("priceText"):setText(g_i18n:formatMoney(price))
    end
		if vehicle.paint ~= nil then
			self:setStatusBarValue(vehList:getAttribute("paintConditionBar"), 1 - vehicle.paint)
		end
		local paintBar = vehList:getAttribute("paintConditionBar").parent
		local needsPaint = Platform.gameplay.hasVehicleDamage
		if needsPaint then
			needsPaint = vehicle.paint ~= nil
		end
		paintBar:setVisible(needsPaint)
		if vehicle.damage ~= nil then
			self:setStatusBarValue(vehList:getAttribute("conditionBar"), 1 - vehicle.damage)
		end
		local damageBar = vehList:getAttribute("conditionBar").parent
		local needsRepair = Platform.gameplay.hasVehicleDamage
		if needsRepair then
			needsRepair = vehicle.damage ~= nil
		end
		damageBar:setVisible(needsRepair)
  else
    operatingTime = 0
		age = 0
	end
	local operatingMins = operatingTime / 60000
	local operatingHours = operatingMins / 60
	local hours = math.floor(operatingHours)
	local mins = (operatingMins - hours * 60) / 6
	local minutes = math.floor(mins) * 10
	vehList:getAttribute("operatingHoursText"):setText(string.format(g_i18n:getText("shop_operatingTime"), hours, minutes))
	vehList:getAttribute("ageText"):setText(string.format(g_i18n:getText("shop_age"), string.format("%d", age)))
end

function WorkshopStorageScreen.onClickBack(self, _)
	WorkshopScreen:superClass().onClickBack(self)

  vsDebug("WorkshopStorageScreen:onClickBack")

	self.vehicle = nil
	self.vehicles = {}
  g_workshopStorageScreen.currentShopLocation = nil
	self:changeScreen(nil)
end

function WorkshopStorageScreen.onClickSpawn(self)

  vsDebug("WorkshopStorageScreen:onClickSpawn")

	if self.vehicle == nil then
		return false
	end
	local text = g_i18n:getText("ui_spawnDialog")
	local callback = self.onYesNoSpawnDialog
	local confirmSound = GuiSoundPlayer.SOUND_SAMPLES.CONFIG_WRENCH
	YesNoDialog.show(callback, self, text, nil, nil, nil, nil, confirmSound)
	return true
end

function WorkshopStorageScreen.onYesNoSpawnDialog(self, confirm)

  vsDebug("WorkshopStorageScreen:onYesNoSpawnDialog")

	if confirm then
    if self.vehicle ~= nil then
      -- Let's spawn the vehicle then delete the file from the storage folder.
      local playerFarmId = 0
      if g_currentMission.missionDynamicInfo.isMultiplayer then
        playerFarmId = g_localPlayer:getFarmId()
      end

      -- Make sure the spawn location is set first
      if self.currentShopLocation ~= nil then
        VehicleSpawnEvent.sendEvent(self.vehicle.vehicleXmlFile, playerFarmId, g_workshopStorageScreen.currentShopLocation,g_localPlayer.userId)
      else 
        ShowDialogEvent.sendEvent(g_localPlayer.userId,g_i18n:getText("info_spawnError"),DialogElement.TYPE_WARNING)
      end

      -- Remove from the table          
      for _, vehicle in pairs(self.storedVehicles) do
        if self.vehicle.uniqueId == vehicle.uniqueId then
          table.removeElement(self.storedVehicles, vehicle)
        end
      end

      -- Reload the vehicles table
      VehicleReloadEvent.sendEvent(playerFarmId)

    end
	end
end


function WorkshopStorageScreen:updateShopLocation(currentShopLocation)
  vsDebug("WorkshopStorageScreen:updateShopLocation")
  vsDebug("currentShopLocation")
  vsDebug(currentShopLocation)

  -- if mp then we need to create an event for the spawn location
  self.currentShopLocation = currentShopLocation
end

function WorkshopStorageScreen.getResetPlaces(self)
  vsDebug("WorkshopStorageScreen:getResetPlaces")

  local wssShopLocation = g_workshopStorageScreen.currentShopLocation
  local newSpawnLocations = {}

  -- Player shop location
  if wssShopLocation ~= nil then
    for _, shopLocation in pairs(wssShopLocation) do
      table.insert(newSpawnLocations, shopLocation)
    end
  end
  
  -- Default shop locations
  if #self.loadSpawnPlaces > 0 then
    for _, shopLocation in pairs(self.loadSpawnPlaces) do
      table.insert(newSpawnLocations, shopLocation)
    end
  else
    for _, shopLocation in pairs(self.storeSpawnPlaces) do
      table.insert(newSpawnLocations, shopLocation)
    end
  end

  vsDebug("New Spawn Locations")
  vsDebug(newSpawnLocations)

  return newSpawnLocations

end

function WorkshopStorageScreen:clearVehicleList()
  vsDebug("WorkshopStorageScreen:clearVehicleList")
  self.storedVehicles = {}
end

function WorkshopStorageScreen:getImageFilename(configFileName)
	local storeItem = g_storeManager:getItemByXMLFilename(configFileName)
	if storeItem == nil then
		return ShopController.EMPTY_FILENAME
	end
	return storeItem.imageFilename
end

function WorkshopStorageScreen:getName(configFileName)
	local storeItem = g_storeManager:getItemByXMLFilename(configFileName)
	if storeItem == nil then
		return "Unknown"
	end
	return storeItem.name
end