--- Script para permitir que la IA deforme el suelo.
--- @author KCHARRO
--- @version 1.0 - Acivar Deformación del Suelo a la IA.
--- @version 2.0 - Reesritura completa para compatibilidad con FS25 1.10.0.0
--- @version 2.1 - Procesamiento recursivo de implementos.
--- @version 2.2 - Solución definitiva para implementos.

local AIGroundDeformation = {}

-- CONFIGURACIÓN AVANZADA
local SETTINGS = {
    DEBUG = false,                  -- Activar para ver mensajes de diagnóstico
    DEFORMATION_SCALE = 1.0,        -- Intensidad de deformación (1.0 = normal)
    FORCE_ALL_AI = true,            -- Forzar incluso en modos no estándar
    CHECK_INTERVAL = 150,           -- Cada 150ms (reducir si hay retraso)
    ENABLE_IMPLEMENT_DEFORM = true  -- Deformación para implementos
}

-- Sistema de traducción
local translations = {
    activation = {
        en = "[True AI Tracks] Script loaded: Ground deformation for AI activated!",
        es = "[Huellas IA] Script cargado: ¡Deformación del suelo para IA activado!",
        de = "[KI-Tracks] Skript geladen: Bodenverformung für KI aktiviert!"
    },
    settings = {
        en = "• Intensity: %.1f | Interval: %dms",
        es = "• Intensidad: %.1f | Intervalo: %dms",
        de = "• Intensität: %.1f | Intervall: %dms"
    },
    debugWheel = {
        en = "[AI Tracks] Wheel processed: %s (Scale: %.1f)",
        es = "[Huellas IA] Rueda procesada: %s (Escala: %.1f)",
        de = "[KI-Tracks] Rad verarbeitet: %s (Skala: %.1f)"
    },
    debugImplement = {
        en = "[AI Tracks] Processing implement: %s",
        es = "[Huellas IA] Procesando implemento: %s",
        de = "[KI-Tracks] Verarbeite Gerät: %s"
    }
}

-- Función para obtener mensaje traducido
local function getMessage(messageKey, ...)
    local lang = g_currentMission and g_currentMission.languageShort or "en"
    local translation = translations[messageKey][lang] or translations[messageKey]["en"]
    return string.format(translation, ...)
end

-- Función auxiliar para verificar IA
local function isAIVehicle(vehicle)
    if not vehicle then return false end
    
    -- Verificar si es un vehículo IA principal
    if vehicle.isAI 
        or (vehicle.spec_aiJobVehicle ~= nil)
        or (vehicle.typeName and vehicle.typeName:match("AI"))
        or (vehicle.cp ~= nil)  -- CoursePlay
        or (vehicle.ad ~= nil)  -- AutoDrive
    then
        return true
    end
    
    -- Verificar si está conectado a un vehículo IA
    if vehicle.spec_attachable then
        local attacherVehicle = vehicle.spec_attachable.attacherVehicle
        if attacherVehicle and isAIVehicle(attacherVehicle) then
            return true
        end
    end
    
    return false
end

-- Variables internas
local lastCheckTime = 0
local processedWheels = {} -- Tabla para evitar procesamiento duplicado

-- Sobrescritura robusta de funciones
local originalSetDisplacementAllowed = WheelPhysics.setDisplacementAllowed
local originalSetDisplacementCollisionEnabled = WheelPhysics.setDisplacementCollisionEnabled

function WheelPhysics:setDisplacementAllowed(displacementAllowed)
    if self.vehicle and isAIVehicle(self.vehicle) then
        displacementAllowed = true
        self.displacementScale = SETTINGS.DEFORMATION_SCALE * 1.2  -- 20% extra para IA
        
        -- Forzar siempre colisión de deformación
        if self.setDisplacementCollisionEnabled then
            self:setDisplacementCollisionEnabled(true)
        end
    end
    
    originalSetDisplacementAllowed(self, displacementAllowed)
    
    if SETTINGS.DEBUG and self.vehicle then
        local vehName = self.vehicle.getName and self.vehicle:getName() or "unknown"
        print(string.format("[AI Tracks] %s: DEFORM %s (Scale: %.1f)", 
              vehName, 
              displacementAllowed and "ON" or "OFF", 
              self.displacementScale or 0))
    end
end

function WheelPhysics:setDisplacementCollisionEnabled(displacementCollisionEnabled)
    if self.vehicle and isAIVehicle(self.vehicle) then
        displacementCollisionEnabled = true
    end
    
    originalSetDisplacementCollisionEnabled(self, displacementCollisionEnabled)
end

-- Verificación activa con seguridad
function AIGroundDeformation:update(dt)
    if not g_currentMission or not g_currentMission.vehicles then return end
    
    lastCheckTime = lastCheckTime + (dt or 0.016)
    if lastCheckTime < (SETTINGS.CHECK_INTERVAL / 1000) then return end
    lastCheckTime = 0

    -- Limpiar tabla de procesados periódicamente
    processedWheels = {}
    
    for _, vehicle in ipairs(g_currentMission.vehicles) do
        if isAIVehicle(vehicle) then
            self:processVehicle(vehicle)
        end
    end
end

-- Procesamiento con validación
function AIGroundDeformation:processVehicle(vehicle)
    if not vehicle then return end
    
    -- Procesar ruedas del vehículo principal
    if vehicle.getWheels then
        for _, wheel in ipairs(vehicle:getWheels()) do
            self:processWheel(wheel)
        end
    end

    -- Procesar implementos recursivamente
    if SETTINGS.ENABLE_IMPLEMENT_DEFORM and vehicle.getAttachedImplements then
        for _, implement in ipairs(vehicle:getAttachedImplements()) do
            if implement.object then
                if SETTINGS.DEBUG then
                    local implName = implement.object.getName and implement.object:getName() or "unknown"
                    print(getMessage("debugImplement", implName))
                end
                self:processVehicle(implement.object)
            end
        end
    end
    
    -- Procesar ruedas en componentes especiales
    if vehicle.spec_wheels then
        for _, wheel in pairs(vehicle.spec_wheels.wheels) do
            self:processWheel(wheel)
        end
    end
end

-- Función para procesar una rueda individual
function AIGroundDeformation:processWheel(wheel)
    if not wheel or not wheel.physics or processedWheels[wheel] then 
        return 
    end
    
    processedWheels[wheel] = true
    
    local physics = wheel.physics
    
    -- Verificar si el vehículo padre es IA
    local isAI = wheel.vehicle and isAIVehicle(wheel.vehicle)
    
    if not isAI and wheel.vehicle and wheel.vehicle.spec_attachable then
        -- Verificar si está conectado a un vehículo IA
        local attacher = wheel.vehicle.spec_attachable.attacherVehicle
        isAI = attacher and isAIVehicle(attacher)
    end
    
    if isAI then
        if physics.setDisplacementAllowed then
            physics:setDisplacementAllowed(true)
        end
        if physics.setDisplacementCollisionEnabled then
            physics:setDisplacementCollisionEnabled(true)
        end
        physics.displacementScale = SETTINGS.DEFORMATION_SCALE * 1.2
        
        if SETTINGS.DEBUG then
            local wheelName = wheel.name or "unknown"
            local vehicleName = wheel.vehicle.getName and wheel.vehicle:getName() or "unknown"
            print(getMessage("debugWheel", wheelName .. " (" .. vehicleName .. ")", physics.displacementScale))
        end
    end
end

-- Inicialización segura
function AIGroundDeformation:loadMap()
    print(getMessage("activation"))
    if SETTINGS.DEBUG then
        print(string.format("• Intensity: %.1f | Interval: %dms", 
              SETTINGS.DEFORMATION_SCALE, SETTINGS.CHECK_INTERVAL))
    end
    
    -- Forzar deformación en IA existentes al cargar
    if g_currentMission and g_currentMission.vehicles then
        for _, vehicle in ipairs(g_currentMission.vehicles) do
            if isAIVehicle(vehicle) then
                self:processVehicle(vehicle)
            end
        end
    end
end

-- Registrar nuevos vehículos
function AIGroundDeformation:onVehicleLoad(vehicle)
    if vehicle and isAIVehicle(vehicle) then
        self:processVehicle(vehicle)
    end
end

-- Registrar eventos
addModEventListener(AIGroundDeformation)

return AIGroundDeformation