--[[
    EncryptionUtil

    Utility class for string encryption and decryption.

    @author: 		BayernGamers
    @date: 			09.04.2025
    @version:		1.0

    History:		v1.0 @09.04.2025 - initial implementation in FS 22
                    ------------------------------------------------------------------------------------------------------
    
    License:        Terms:
                        Usage:
                            Feel free to use this work as-is as long as you adhere to the following terms:
                        Attribution:
                            You must give appropriate credit to the original author when using this work.
                        No Derivatives:
                            You may not alter, transform, or build upon this work in any way.
                        Usage: 
                            The work may be used for personal and commercial purposes, provided it is not modified or adapted.
                        Additional Clause:
                            This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]
source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))
EncryptionUtil = {}

local log = LoggingUtil.new(true, LoggingUtil.DEBUG_LEVELS.HIGH, "EncryptionUtil.lua")

function EncryptionUtil.numericBitXOR (a,b)
    local p,c=1,0
    while a>0 and b>0 do
        local ra,rb=a%2,b%2
        if ra~=rb then c=c+p end
        a,b,p=(a-ra)/2,(b-rb)/2,p*2
    end
    if a<b then a=b end
    while a>0 do
        local ra=a%2
        if ra>0 then c=c+p end
        a,p=(a-ra)/2,p*2
    end
    return c
end

function EncryptionUtil.BitXOR (a, b)
    local result = ""
    local maxLength = math.max(#a, #b)
    a = string.rep("\0", maxLength - #a) .. a
    b = string.rep("\0", maxLength - #b) .. b

    for i = 1, maxLength do
        local charA, charB = string.byte(a, i), string.byte(b, i)
        local xorResult = SecureMod.numericBitXOR(charA, charB)
        result = result .. string.char(xorResult)
    end

    return result
end

function EncryptionUtil.generateIV(length)
    local iv = {}
    for i = 1, length do
        table.insert(iv, string.char(math.random(0, 255)))
    end
    return table.concat(iv)
end

function EncryptionUtil.generateIV(length)
    local iv = {}
    for i = 1, length do
        table.insert(iv, string.char(math.random(0, 255)))
    end
    return table.concat(iv)
end

-- Hilfsfunktion: Konvertiere Binärdaten in einen HEX-String
function EncryptionUtil.toHex(data)
    return (data:gsub('.', function(byte)
        return string.format('%02X', string.byte(byte))
    end))
end

-- Hilfsfunktion: Konvertiere einen HEX-String zurück in Binärdaten
function EncryptionUtil.fromHex(hex)
    return (hex:gsub('..', function(byte)
        return string.char(tonumber(byte, 16))
    end))
end

function EncryptionUtil.encrypt(input, password)
    local iv = EncryptionUtil.generateIV(16) -- Initialisierungsvektor für die Daten
    local passwordIV = EncryptionUtil.generateIV(16) -- Initialisierungsvektor für das Passwort
    local hashedPassword = getMD5(password .. passwordIV) -- Passwort mit IV hashen
    local result = {iv, passwordIV} -- IVs an den Anfang der verschlüsselten Nachricht setzen
    local passwordLength = #hashedPassword

    for i = 1, #input do
        local inputChar = string.byte(input, i)
        local passwordChar = string.byte(hashedPassword, (i - 1) % passwordLength + 1)
        local ivChar = string.byte(iv, (i - 1) % #iv + 1)

        -- XOR mit Passwort und IV
        local xorResult = EncryptionUtil.numericBitXOR(inputChar, passwordChar)
        xorResult = EncryptionUtil.numericBitXOR(xorResult, ivChar)

        table.insert(result, string.char(xorResult))
    end

    -- Rückgabe des verschlüsselten Strings (IV + Passwort-IV + verschlüsselte Daten) als HEX-String
    local resultString = table.concat(result)
    return EncryptionUtil.toHex(resultString)
end

function EncryptionUtil.decrypt(encrypted, password)
    -- Konvertiere den HEX-String zurück in Binärdaten
    local decoded = EncryptionUtil.fromHex(encrypted)

    local iv = decoded:sub(1, 16) -- Der IV ist die ersten 16 Bytes der verschlüsselten Nachricht
    local passwordIV = decoded:sub(17, 32) -- Der Passwort-IV sind die nächsten 16 Bytes
    local encryptedData = decoded:sub(33)
    local hashedPassword = getMD5(password .. passwordIV) -- Passwort mit IV hashen
    local result = {}
    local passwordLength = #hashedPassword

    for i = 1, #encryptedData do
        local encryptedChar = string.byte(encryptedData, i)
        local passwordChar = string.byte(hashedPassword, (i - 1) % passwordLength + 1)
        local ivChar = string.byte(iv, (i - 1) % #iv + 1)

        -- XOR mit Passwort und IV rückgängig machen
        local xorResult = EncryptionUtil.numericBitXOR(encryptedChar, ivChar)
        local originalChar = EncryptionUtil.numericBitXOR(xorResult, passwordChar)

        table.insert(result, string.char(originalChar))
    end

    -- Rückgabe des entschlüsselten Strings
    return table.concat(result)
end